import express from 'express';
import { User, File, ActivityLog, ShareLink } from '../models/index.js';
import { authenticate, authorize } from '../middleware/auth.js';
import { validateUserUpdate, validateUUID } from '../middleware/validation.js';
import { logUserActivity } from '../middleware/logging.js';
import { AppError, NotFoundError } from '../middleware/errorHandler.js';
import { logger } from '../utils/logger.js';
import { Op } from 'sequelize';
import bcrypt from 'bcryptjs';

const router = express.Router();

// All routes require admin authentication
router.use(authenticate);
router.use(authorize('admin'));

// Get system statistics
router.get('/stats', async (req, res, next) => {
  try {
    const { period = '30d' } = req.query;
    
    // Calculate date range
    const now = new Date();
    const startDate = new Date();
    
    switch (period) {
      case '7d':
        startDate.setDate(now.getDate() - 7);
        break;
      case '30d':
        startDate.setDate(now.getDate() - 30);
        break;
      case '90d':
        startDate.setDate(now.getDate() - 90);
        break;
      case '1y':
        startDate.setFullYear(now.getFullYear() - 1);
        break;
      default:
        startDate.setDate(now.getDate() - 30);
    }

    // Get user statistics
    const totalUsers = await User.count();
    const activeUsers = await User.count({ where: { is_active: true } });
    const newUsers = await User.count({
      where: {
        created_at: { [Op.gte]: startDate }
      }
    });

    // Get file statistics
    const totalFiles = await File.count({ where: { type: 'file', is_deleted: false } });
    const totalFolders = await File.count({ where: { type: 'folder', is_deleted: false } });
    const deletedFiles = await File.count({ where: { is_deleted: true } });

    // Get storage statistics
    const storageStats = await User.findAll({
      attributes: [
        [User.sequelize.fn('SUM', User.sequelize.col('storage_used')), 'totalUsed'],
        [User.sequelize.fn('SUM', User.sequelize.col('storage_quota')), 'totalQuota']
      ]
    });

    const totalStorageUsed = parseInt(storageStats[0]?.dataValues?.totalUsed || 0);
    const totalStorageQuota = parseInt(storageStats[0]?.dataValues?.totalQuota || 0);

    // Get activity statistics
    const recentActivity = await ActivityLog.count({
      where: {
        created_at: { [Op.gte]: startDate }
      }
    });

    // Get share statistics
    const activeShares = await ShareLink.count({ where: { is_active: true } });

    res.json({
      users: {
        total: totalUsers,
        active: activeUsers,
        new: newUsers,
        inactive: totalUsers - activeUsers
      },
      files: {
        total: totalFiles,
        folders: totalFolders,
        deleted: deletedFiles
      },
      storage: {
        used: totalStorageUsed,
        quota: totalStorageQuota,
        percentage: totalStorageQuota > 0 ? (totalStorageUsed / totalStorageQuota) * 100 : 0
      },
      activity: {
        recent: recentActivity
      },
      shares: {
        active: activeShares
      },
      period
    });
  } catch (error) {
    next(error);
  }
});

// Get all users
router.get('/users', async (req, res, next) => {
  try {
    const { 
      page = 1, 
      limit = 20, 
      search, 
      role, 
      status = 'all',
      sortBy = 'created_at',
      sortOrder = 'DESC'
    } = req.query;

    const offset = (parseInt(page) - 1) * parseInt(limit);
    const where = {};

    // Apply filters
    if (search) {
      where[Op.or] = [
        { name: { [Op.iLike]: `%${search}%` } },
        { email: { [Op.iLike]: `%${search}%` } }
      ];
    }

    if (role && role !== 'all') {
      where.role = role;
    }

    if (status !== 'all') {
      where.is_active = status === 'active';
    }

    const { count, rows: users } = await User.findAndCountAll({
      where,
      limit: parseInt(limit),
      offset,
      order: [[sortBy, sortOrder.toUpperCase()]],
      include: [
        {
          model: File,
          as: 'files',
          where: { is_deleted: false },
          required: false,
          attributes: ['id', 'type', 'size']
        }
      ]
    });

    // Add computed fields
    const usersWithStats = users.map(user => {
      const files = user.files || [];
      return {
        ...user.toJSON(),
        storageUsed: user.storage_used,
        storageQuota: user.storage_quota,
        stats: {
          totalFiles: files.filter(f => f.type === 'file').length,
          totalFolders: files.filter(f => f.type === 'folder').length,
          storagePercentage: (user.storage_used / user.storage_quota) * 100
        }
      };
    });

    res.json({
      users: usersWithStats,
      pagination: {
        total: count,
        page: parseInt(page),
        limit: parseInt(limit),
        totalPages: Math.ceil(count / parseInt(limit)),
        hasMore: offset + parseInt(limit) < count
      }
    });
  } catch (error) {
    next(error);
  }
});

// Get user details
router.get('/users/:id', validateUUID('id'), async (req, res, next) => {
  try {
    const user = await User.findByPk(req.params.id, {
      include: [
        {
          model: File,
          as: 'files',
          where: { is_deleted: false },
          required: false
        },
        {
          model: ActivityLog,
          as: 'activities',
          limit: 10,
          order: [['created_at', 'DESC']]
        }
      ]
    });

    if (!user) {
      throw new NotFoundError('User');
    }

    res.json({ user });
  } catch (error) {
    next(error);
  }
});

// Update user
router.patch('/users/:id', validateUUID('id'), validateUserUpdate, logUserActivity('admin_update_user'), async (req, res, next) => {
  try {
    const { name, email, role, storageQuota, isActive } = req.body;

    const user = await User.findByPk(req.params.id);
    if (!user) {
      throw new NotFoundError('User');
    }

    // Prevent admin from demoting themselves
    if (user.id === req.user.id && role && role !== 'admin') {
      throw new AppError('Cannot change your own role', 400);
    }

    // Update fields
    if (name) user.name = name;
    if (email) user.email = email.toLowerCase();
    if (role) user.role = role;
    if (storageQuota !== undefined) user.storage_quota = storageQuota;
    if (isActive !== undefined) user.is_active = isActive;

    await user.save();

    logger.info(`User updated by admin: ${user.email} by ${req.user.email}`);

    res.json({
      message: 'User updated successfully',
      user: user.toJSON()
    });
  } catch (error) {
    next(error);
  }
});

// Delete user
router.delete('/users/:id', validateUUID('id'), logUserActivity('admin_delete_user'), async (req, res, next) => {
  try {
    const user = await User.findByPk(req.params.id);
    if (!user) {
      throw new NotFoundError('User');
    }

    // Prevent admin from deleting themselves
    if (user.id === req.user.id) {
      throw new AppError('Cannot delete your own account', 400);
    }

    // Soft delete user
    user.is_active = false;
    await user.save();

    logger.info(`User deleted by admin: ${user.email} by ${req.user.email}`);

    res.json({
      message: 'User deleted successfully'
    });
  } catch (error) {
    next(error);
  }
});

// Create a new user (admin)
router.post('/users', async (req, res, next) => {
  try {
    const { name, email, password, role = 'user', storageQuota = 5 * 1024 * 1024 * 1024, isActive = true } = req.body;
    if (!name || !email || !password) {
      throw new AppError('Name, email, and password are required', 400);
    }
    // Prevent duplicate email
    const existing = await User.findOne({ where: { email: email.toLowerCase() } });
    if (existing) {
      throw new AppError('Email already exists', 400);
    }
    const user = await User.create({
      name,
      email: email.toLowerCase(),
      password,
      role,
      storage_quota: storageQuota,
      is_active: isActive
    });
    logger.info(`User created by admin: ${user.email} by ${req.user.email}`);
    res.status(201).json({
      message: 'User created successfully',
      user: user.toJSON()
    });
  } catch (error) {
    next(error);
  }
});

// Get system activity logs
router.get('/activity', async (req, res, next) => {
  try {
    const { 
      page = 1, 
      limit = 50, 
      userId, 
      action, 
      resourceType,
      startDate,
      endDate
    } = req.query;

    const offset = (parseInt(page) - 1) * parseInt(limit);
    const where = {};

    // Apply filters
    if (userId) where.user_id = userId;
    if (action) where.action = action;
    if (resourceType) where.resource_type = resourceType;

    if (startDate || endDate) {
      where.created_at = {};
      if (startDate) where.created_at[Op.gte] = new Date(startDate);
      if (endDate) where.created_at[Op.lte] = new Date(endDate);
    }

    const { count, rows: activities } = await ActivityLog.findAndCountAll({
      where,
      limit: parseInt(limit),
      offset,
      order: [['created_at', 'DESC']],
      include: [
        {
          model: User,
          as: 'user',
          attributes: ['id', 'name', 'email']
        }
      ]
    });

    res.json({
      activities,
      pagination: {
        total: count,
        page: parseInt(page),
        limit: parseInt(limit),
        totalPages: Math.ceil(count / parseInt(limit)),
        hasMore: offset + parseInt(limit) < count
      }
    });
  } catch (error) {
    next(error);
  }
});

// Get activity statistics
router.get('/activity/stats', async (req, res, next) => {
  try {
    const { period = '30d' } = req.query;
    
    const now = new Date();
    const startDate = new Date();
    
    switch (period) {
      case '7d':
        startDate.setDate(now.getDate() - 7);
        break;
      case '30d':
        startDate.setDate(now.getDate() - 30);
        break;
      case '90d':
        startDate.setDate(now.getDate() - 90);
        break;
      default:
        startDate.setDate(now.getDate() - 30);
    }

    const stats = await ActivityLog.getStats(startDate, now);

    res.json({ stats, period });
  } catch (error) {
    next(error);
  }
});

// Get all files (admin view)
router.get('/files', async (req, res, next) => {
  try {
    const { 
      page = 1, 
      limit = 20, 
      search, 
      type, 
      ownerId,
      sortBy = 'created_at',
      sortOrder = 'DESC'
    } = req.query;

    const offset = (parseInt(page) - 1) * parseInt(limit);
    const where = { is_deleted: false };

    // Apply filters
    if (search) {
      where.name = { [Op.iLike]: `%${search}%` };
    }

    if (type && type !== 'all') {
      where.type = type;
    }

    if (ownerId) {
      where.owner_id = ownerId;
    }

    const { count, rows: files } = await File.findAndCountAll({
      where,
      limit: parseInt(limit),
      offset,
      order: [[sortBy, sortOrder.toUpperCase()]],
      include: [
        {
          model: User,
          as: 'owner',
          attributes: ['id', 'name', 'email']
        }
      ]
    });

    res.json({
      files,
      pagination: {
        total: count,
        page: parseInt(page),
        limit: parseInt(limit),
        totalPages: Math.ceil(count / parseInt(limit)),
        hasMore: offset + parseInt(limit) < count
      }
    });
  } catch (error) {
    next(error);
  }
});

// System maintenance endpoints
router.post('/maintenance/cleanup-expired-sessions', async (req, res, next) => {
  try {
    const cleanedCount = await UserSession.cleanupExpired();
    
    logger.info(`Cleaned up ${cleanedCount} expired sessions`);
    
    res.json({
      message: 'Expired sessions cleaned up',
      cleanedCount
    });
  } catch (error) {
    next(error);
  }
});

router.post('/maintenance/cleanup-expired-shares', async (req, res, next) => {
  try {
    const cleanedCount = await ShareLink.cleanupExpired();
    
    logger.info(`Cleaned up ${cleanedCount} expired share links`);
    
    res.json({
      message: 'Expired share links cleaned up',
      cleanedCount
    });
  } catch (error) {
    next(error);
  }
});

router.post('/maintenance/cleanup-expired-notifications', async (req, res, next) => {
  try {
    const cleanedCount = await Notification.cleanupExpired();
    
    logger.info(`Cleaned up ${cleanedCount} expired notifications`);
    
    res.json({
      message: 'Expired notifications cleaned up',
      cleanedCount
    });
  } catch (error) {
    next(error);
  }
});

// Reset a user's storage usage and quota (admin only)
router.post('/user/:userId/reset-storage', async (req, res, next) => {
  try {
    const { userId } = req.params;
    const { storage_used = 0, storage_quota = 10 * 1024 * 1024 * 1024 } = req.body; // default: 0 used, 10GB quota
    const user = await User.findByPk(userId);
    if (!user) return res.status(404).json({ error: 'User not found' });
    user.storage_used = storage_used;
    user.storage_quota = storage_quota;
    await user.save();
    res.json({ message: 'User storage reset', user: { id: user.id, storage_used: user.storage_used, storage_quota: user.storage_quota } });
  } catch (error) {
    next(error);
  }
});

// Admin: Reset user password (send reset email)
router.post('/users/:id/reset-password', validateUUID('id'), async (req, res, next) => {
  try {
    const user = await User.findByPk(req.params.id);
    if (!user) throw new NotFoundError('User');
    // Generate a reset token and expiry
    const token = Math.random().toString(36).slice(2) + Date.now().toString(36);
    user.reset_token = token;
    user.reset_token_expires = new Date(Date.now() + 1000 * 60 * 60); // 1 hour
    await user.save();
    // Send email (replace with your email logic)
    // await sendResetEmail(user.email, token);
    logger.info(`Admin triggered password reset for ${user.email}`);
    res.json({ message: 'Password reset email sent (simulated)', token });
  } catch (error) {
    next(error);
  }
});

// Admin: Set a new password for a user
router.post('/users/:id/set-password', validateUUID('id'), async (req, res, next) => {
  try {
    const { password } = req.body;
    if (!password || password.length < 6) {
      throw new AppError('Password must be at least 6 characters', 400);
    }
    const user = await User.findByPk(req.params.id);
    if (!user) throw new NotFoundError('User');
    // Assign plain password, let model hook hash it
    user.password = password;
    await user.save();
    logger.info(`Admin set new password for ${user.email}`);
    res.json({ message: 'Password updated successfully' });
  } catch (error) {
    next(error);
  }
});

export default router;