import { User } from './User.js';
import { File } from './File.js';
import { FileVersion } from './FileVersion.js';
import { ShareLink } from './ShareLink.js';
import { ActivityLog } from './ActivityLog.js';
import { Notification } from './Notification.js';
import { UserSession } from './UserSession.js';
import { Trash } from './Trash.js';
import { FileShare } from './FileShare.js';

// Define associations
export function defineAssociations() {
  // User associations
  User.hasMany(File, { foreignKey: 'owner_id', as: 'files' });
  User.hasMany(ActivityLog, { foreignKey: 'user_id', as: 'activities' });
  User.hasMany(Notification, { foreignKey: 'user_id', as: 'notifications' });
  User.hasMany(UserSession, { foreignKey: 'user_id', as: 'sessions' });
  User.hasMany(ShareLink, { foreignKey: 'created_by', as: 'sharedLinks' });
  User.hasMany(Trash, { foreignKey: 'owner_id', as: 'trash' });
  User.hasMany(FileShare, { foreignKey: 'user_id', as: 'fileShares' });

  // File associations
  File.belongsTo(User, { foreignKey: 'owner_id', as: 'owner' });
  File.hasMany(File, { foreignKey: 'parent_id', as: 'children' });
  File.belongsTo(File, { foreignKey: 'parent_id', as: 'parent' });
  File.hasMany(FileVersion, { foreignKey: 'file_id', as: 'versions' });
  File.hasMany(ShareLink, { foreignKey: 'file_id', as: 'shareLinks' });
  File.hasMany(ActivityLog, { foreignKey: 'resource_id', as: 'activities' });
  File.hasOne(Trash, { foreignKey: 'file_id', as: 'trash' });
  File.hasMany(FileShare, { foreignKey: 'file_id', as: 'shares' });

  // FileVersion associations
  FileVersion.belongsTo(File, { foreignKey: 'file_id', as: 'file' });
  FileVersion.belongsTo(User, { foreignKey: 'created_by', as: 'creator' });

  // ShareLink associations
  ShareLink.belongsTo(File, { foreignKey: 'file_id', as: 'file' });
  ShareLink.belongsTo(User, { foreignKey: 'created_by', as: 'creator' });

  // ActivityLog associations
  ActivityLog.belongsTo(User, { foreignKey: 'user_id', as: 'user' });

  // Notification associations
  Notification.belongsTo(User, { foreignKey: 'user_id', as: 'user' });

  // UserSession associations
  UserSession.belongsTo(User, { foreignKey: 'user_id', as: 'user' });

  // Trash associations
  Trash.belongsTo(File, { foreignKey: 'file_id', as: 'file' });
  Trash.belongsTo(User, { foreignKey: 'owner_id', as: 'owner' });

  // FileShare associations
  FileShare.belongsTo(File, { foreignKey: 'file_id' });
  FileShare.belongsTo(User, { foreignKey: 'user_id' });
}

export async function initializeDatabase() {
  defineAssociations();
  
  // Create default admin user if it doesn't exist
  const adminEmail = process.env.ADMIN_EMAIL || 'admin@cloudstorage.com';
  const existingAdmin = await User.findOne({ where: { email: adminEmail } });
  
  if (!existingAdmin) {
    await User.create({
      email: adminEmail,
      name: 'System Administrator',
      password: 'admin123', // This will be hashed by the model
      role: 'admin',
      storage_quota: 100 * 1024 * 1024 * 1024, // 100GB
      is_verified: true
    });
    console.log(`Default admin user created: ${adminEmail}`);
  }
}

export {
  User,
  File,
  FileVersion,
  ShareLink,
  ActivityLog,
  Notification,
  UserSession,
  Trash,
  FileShare
};