import { DataTypes } from 'sequelize';
import { sequelize } from '../config/database.js';

export const ActivityLog = sequelize.define('activity_logs', {
  id: {
    type: DataTypes.UUID,
    defaultValue: DataTypes.UUIDV4,
    primaryKey: true
  },
  user_id: {
    type: DataTypes.UUID,
    allowNull: false,
    references: {
      model: 'users',
      key: 'id'
    }
  },
  action: {
    type: DataTypes.STRING,
    allowNull: false
  },
  resource_type: {
    type: DataTypes.STRING,
    allowNull: false
  },
  resource_id: {
    type: DataTypes.UUID,
    allowNull: true
  },
  resource_name: {
    type: DataTypes.STRING,
    allowNull: true
  },
  details: {
    type: DataTypes.JSONB,
    defaultValue: {}
  },
  ip_address: {
    type: DataTypes.INET,
    allowNull: true
  },
  user_agent: {
    type: DataTypes.TEXT,
    allowNull: true
  }
}, {
  indexes: [
    {
      fields: ['user_id']
    },
    {
      fields: ['action']
    },
    {
      fields: ['resource_type']
    },
    {
      fields: ['resource_id']
    },
    {
      fields: ['created_at']
    }
  ]
});

// Class methods
ActivityLog.logActivity = async function(data) {
  return this.create({
    user_id: data.userId,
    action: data.action,
    resource_type: data.resourceType,
    resource_id: data.resourceId,
    resource_name: data.resourceName,
    details: data.details || {},
    ip_address: data.ipAddress,
    user_agent: data.userAgent
  });
};

ActivityLog.findByUser = function(userId, limit = 50) {
  return this.findAll({
    where: { user_id: userId },
    order: [['created_at', 'DESC']],
    limit,
    include: ['user']
  });
};

ActivityLog.findRecent = function(limit = 100) {
  return this.findAll({
    order: [['created_at', 'DESC']],
    limit,
    include: ['user']
  });
};

ActivityLog.findByResource = function(resourceType, resourceId) {
  return this.findAll({
    where: {
      resource_type: resourceType,
      resource_id: resourceId
    },
    order: [['created_at', 'DESC']],
    include: ['user']
  });
};

ActivityLog.getStats = async function(startDate, endDate) {
  const where = {};
  if (startDate && endDate) {
    where.created_at = {
      [sequelize.Sequelize.Op.between]: [startDate, endDate]
    };
  }
  
  const stats = await this.findAll({
    where,
    attributes: [
      'action',
      [sequelize.fn('COUNT', sequelize.col('id')), 'count']
    ],
    group: ['action'],
    order: [[sequelize.fn('COUNT', sequelize.col('id')), 'DESC']]
  });
  
  return stats;
};