import { body, param, query, validationResult } from 'express-validator';

export const handleValidationErrors = (req, res, next) => {
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).json({
      error: 'Validation failed',
      details: errors.array()
    });
  }
  next();
};

// Auth validation rules
export const validateRegister = [
  body('email')
    .isEmail()
    .normalizeEmail({ gmail_remove_dots: false })
    .withMessage('Valid email is required'),
  body('password')
    .isLength({ min: 6 })
    .withMessage('Password must be at least 6 characters long'),
  body('name')
    .trim()
    .isLength({ min: 2, max: 100 })
    .withMessage('Name must be between 2 and 100 characters'),
  handleValidationErrors
];

export const validateLogin = [
  body('email')
    .isEmail()
    .normalizeEmail({ gmail_remove_dots: false })
    .withMessage('Valid email is required'),
  body('password')
    .notEmpty()
    .withMessage('Password is required'),
  handleValidationErrors
];

// File validation rules
export const validateFileUpload = [
  (req, res, next) => {
    // Debug: log incoming body before validation
    console.log('DEBUG validateFileUpload req.body:', req.body);
    if (req.body) {
      Object.entries(req.body).forEach(([k, v]) => {
        console.log(`  ${k}:`, v, 'type:', typeof v);
      });
    }
    next();
  },
  body('name')
    .trim()
    .isLength({ min: 1, max: 255 })
    .withMessage('File name must be between 1 and 255 characters'),
  body('size')
    .isInt({ min: 0, max: 100 * 1024 * 1024 }) // Allow up to 100MB
    .withMessage('File size must be a positive integer and less than 100MB'),
  body('mimeType')
    .optional()
    .isString()
    .withMessage('MIME type must be a string'),
  handleValidationErrors
];

export const validateFolderCreate = [
  body('name')
    .trim()
    .isLength({ min: 1, max: 255 })
    .withMessage('Folder name must be between 1 and 255 characters')
    .matches(/^[^<>:"/\\|?*]+$/)
    .withMessage('Folder name contains invalid characters'),
  body('parentId')
    .optional()
    .isUUID()
    .withMessage('Parent ID must be a valid UUID'),
  handleValidationErrors
];

export const validateFileMove = [
  body('targetParentId')
    .optional()
    .isUUID()
    .withMessage('Target parent ID must be a valid UUID'),
  body('newName')
    .optional()
    .trim()
    .isLength({ min: 1, max: 255 })
    .withMessage('New name must be between 1 and 255 characters'),
  handleValidationErrors
];

// Share validation rules
export const validateShareLink = [
  body('permissions')
    .isArray()
    .withMessage('Permissions must be an array')
    .custom((permissions) => {
      const validPermissions = ['view', 'download', 'edit'];
      return permissions.every(p => validPermissions.includes(p));
    })
    .withMessage('Invalid permissions'),
  body('expiresAt')
    .optional()
    .isISO8601()
    .withMessage('Expiry date must be a valid ISO 8601 date'),
  body('password')
    .optional()
    .isLength({ min: 4 })
    .withMessage('Password must be at least 4 characters long'),
  body('maxDownloads')
    .optional()
    .isInt({ min: 1 })
    .withMessage('Max downloads must be a positive integer'),
  handleValidationErrors
];

// User validation rules
export const validateUserUpdate = [
  body('name')
    .optional()
    .trim()
    .isLength({ min: 2, max: 100 })
    .withMessage('Name must be between 2 and 100 characters'),
  body('email')
    .optional()
    .isEmail()
    .normalizeEmail({ gmail_remove_dots: false })
    .withMessage('Valid email is required'),
  body('role')
    .optional()
    .isIn(['admin', 'user'])
    .withMessage('Role must be either admin or user'),
  body('storageQuota')
    .optional()
    .isInt({ min: 0 })
    .withMessage('Storage quota must be a positive integer'),
  handleValidationErrors
];

// Search validation rules
export const validateSearch = [
  query('q')
    .trim()
    .isLength({ min: 1, max: 100 })
    .withMessage('Search query must be between 1 and 100 characters'),
  query('type')
    .optional()
    .isIn(['file', 'folder', 'all'])
    .withMessage('Type must be file, folder, or all'),
  query('limit')
    .optional()
    .isInt({ min: 1, max: 100 })
    .withMessage('Limit must be between 1 and 100'),
  handleValidationErrors
];

// UUID parameter validation
export const validateUUID = (paramName) => [
  param(paramName)
    .isUUID()
    .withMessage(`${paramName} must be a valid UUID`),
  handleValidationErrors
];