import express from 'express';
import bcrypt from 'bcryptjs';
import { File, ShareLink, User, ActivityLog } from '../models/index.js';
import { authenticate, optionalAuth } from '../middleware/auth.js';
import { validateShareLink, validateUUID } from '../middleware/validation.js';
import { logFileActivity } from '../middleware/logging.js';
import { s3Service } from '../utils/s3.js';
import { AppError, NotFoundError } from '../middleware/errorHandler.js';
import { logger } from '../utils/logger.js';

const router = express.Router();

// Create share link (requires authentication)
router.post('/:fileId', authenticate, validateUUID('fileId'), validateShareLink, logFileActivity('create_share'), async (req, res, next) => {
  try {
    const { permissions, expiresAt, password, maxDownloads } = req.body;
    const fileId = req.params.fileId;

    // Check if file exists and user owns it
    const file = await File.findOne({
      where: {
        id: fileId,
        owner_id: req.user.id,
        is_deleted: false
      }
    });

    if (!file) {
      throw new NotFoundError('File');
    }

    // Hash password if provided
    let hashedPassword = null;
    if (password) {
      hashedPassword = await bcrypt.hash(password, 12);
    }

    // Create share link
    const shareLink = await ShareLink.create({
      file_id: fileId,
      permissions,
      expires_at: expiresAt ? new Date(expiresAt) : null,
      password: hashedPassword,
      max_downloads: maxDownloads,
      created_by: req.user.id
    });

    // Generate S3 presigned download URL
    let s3DownloadUrl = null;
    if (file.s3_key) {
      try {
        s3DownloadUrl = await s3Service.getDownloadUrl(file.s3_key);
      } catch (err) {
        logger.error('Failed to generate S3 download URL:', err);
        s3DownloadUrl = null;
      }
    } else {
      logger.warn(`File ${file.id} does not have a valid s3_key, cannot generate S3 download link.`);
    }

    // Mark file as shared
    if (!file.is_shared) {
      file.is_shared = true;
      await file.save();
    }

    logger.info(`Share link created for file: ${file.name} by user ${req.user.email}`);

    res.status(201).json({
      message: 'Share link created successfully',
      shareLink: {
        id: shareLink.id,
        token: shareLink.token,
        url: shareLink.getPublicUrl(),
        s3DownloadUrl, // <-- Add S3 download URL to response
        permissions: shareLink.permissions,
        expiresAt: shareLink.expires_at,
        hasPassword: !!shareLink.password,
        maxDownloads: shareLink.max_downloads,
        downloadCount: shareLink.download_count,
        createdAt: shareLink.created_at
      }
    });
  } catch (error) {
    next(error);
  }
});

// Get share links for a file
router.get('/file/:fileId', authenticate, validateUUID('fileId'), async (req, res, next) => {
  try {
    const fileId = req.params.fileId;

    // Check if file exists and user owns it
    const file = await File.findOne({
      where: {
        id: fileId,
        owner_id: req.user.id,
        is_deleted: false
      }
    });

    if (!file) {
      throw new NotFoundError('File');
    }

    const shareLinks = await ShareLink.findByFile(fileId);

    const shareLinksWithUrls = shareLinks.map(link => ({
      id: link.id,
      token: link.token,
      url: link.getPublicUrl(),
      permissions: link.permissions,
      expiresAt: link.expires_at,
      hasPassword: !!link.password,
      maxDownloads: link.max_downloads,
      downloadCount: link.download_count,
      accessCount: link.access_count,
      lastAccessed: link.last_accessed,
      isActive: link.is_active,
      canAccess: link.canAccess(),
      createdAt: link.created_at
    }));

    res.json({ shareLinks: shareLinksWithUrls });
  } catch (error) {
    next(error);
  }
});

// Get user's share links
router.get('/my-shares', authenticate, async (req, res, next) => {
  try {
    const { page = 1, limit = 20 } = req.query;
    const offset = (parseInt(page) - 1) * parseInt(limit);

    const shareLinks = await ShareLink.findAll({
      where: { created_by: req.user.id },
      include: [
        {
          model: File,
          as: 'file',
          attributes: ['id', 'name', 'type', 'size', 'mime_type']
        }
      ],
      order: [['created_at', 'DESC']],
      limit: parseInt(limit),
      offset
    });

    const total = await ShareLink.count({
      where: { created_by: req.user.id }
    });

    const shareLinksWithUrls = shareLinks.map(link => ({
      id: link.id,
      token: link.token,
      url: link.getPublicUrl(),
      file: link.file,
      permissions: link.permissions,
      expiresAt: link.expires_at,
      hasPassword: !!link.password,
      maxDownloads: link.max_downloads,
      downloadCount: link.download_count,
      accessCount: link.access_count,
      lastAccessed: link.last_accessed,
      isActive: link.is_active,
      canAccess: link.canAccess(),
      createdAt: link.created_at
    }));

    res.json({
      shareLinks: shareLinksWithUrls,
      pagination: {
        total,
        page: parseInt(page),
        limit: parseInt(limit),
        totalPages: Math.ceil(total / parseInt(limit)),
        hasMore: offset + parseInt(limit) < total
      }
    });
  } catch (error) {
    next(error);
  }
});

// Access shared file (public endpoint)
router.get('/:token', optionalAuth, async (req, res, next) => {
  try {
    const { token } = req.params;
    const { password } = req.query;

    // Find share link
    const shareLink = await ShareLink.findByToken(token);
    if (!shareLink) {
      throw new NotFoundError('Share link');
    }

    // Check if share link can be accessed
    if (!shareLink.canAccess()) {
      throw new AppError('Share link is expired or no longer available', 410);
    }

    // Check password if required
    if (shareLink.password) {
      if (!password) {
        return res.status(401).json({
          error: 'Password required',
          requiresPassword: true
        });
      }

      const isPasswordValid = await bcrypt.compare(password, shareLink.password);
      if (!isPasswordValid) {
        throw new AppError('Invalid password', 401);
      }
    }

    // Record access
    await shareLink.recordAccess();

    // Get file details
    const file = shareLink.file;
    if (!file || file.is_deleted) {
      throw new NotFoundError('File');
    }

    // Log activity
    if (req.user) {
      await ActivityLog.logActivity({
        userId: req.user.id,
        action: 'access_shared_file',
        resourceType: 'file',
        resourceId: file.id,
        resourceName: file.name,
        details: {
          shareToken: token,
          sharedBy: shareLink.created_by
        },
        ipAddress: req.ip,
        userAgent: req.get('User-Agent')
      });
    }

    res.json({
      file: {
        id: file.id,
        name: file.name,
        type: file.type,
        size: file.size,
        mimeType: file.mime_type,
        createdAt: file.created_at,
        modifiedAt: file.updated_at
      },
      shareLink: {
        permissions: shareLink.permissions,
        expiresAt: shareLink.expires_at,
        downloadCount: shareLink.download_count,
        maxDownloads: shareLink.max_downloads
      },
      owner: {
        name: shareLink.creator.name
      }
    });
  } catch (error) {
    next(error);
  }
});

// Download shared file
router.get('/:token/download', optionalAuth, async (req, res, next) => {
  try {
    const { token } = req.params;
    const { password } = req.query;

    // Find share link
    const shareLink = await ShareLink.findByToken(token);
    if (!shareLink) {
      throw new NotFoundError('Share link');
    }

    // Check if share link can be accessed
    if (!shareLink.canAccess()) {
      throw new AppError('Share link is expired or no longer available', 410);
    }

    // Check if download permission is granted
    if (!shareLink.permissions.includes('download')) {
      throw new AppError('Download not permitted', 403);
    }

    // Check password if required
    if (shareLink.password) {
      if (!password) {
        throw new AppError('Password required', 401);
      }

      const isPasswordValid = await bcrypt.compare(password, shareLink.password);
      if (!isPasswordValid) {
        throw new AppError('Invalid password', 401);
      }
    }

    // Get file details
    const file = shareLink.file;
    if (!file || file.is_deleted || file.type !== 'file') {
      throw new NotFoundError('File');
    }

    // Record download
    await shareLink.recordDownload();

    // Generate download URL
    const downloadUrl = await s3Service.getDownloadUrl(file.s3_key);

    // Log activity
    if (req.user) {
      await ActivityLog.logActivity({
        userId: req.user.id,
        action: 'download_shared_file',
        resourceType: 'file',
        resourceId: file.id,
        resourceName: file.name,
        details: {
          shareToken: token,
          sharedBy: shareLink.created_by
        },
        ipAddress: req.ip,
        userAgent: req.get('User-Agent')
      });
    }

    logger.info(`Shared file downloaded: ${file.name} via token ${token}`);

    res.json({
      downloadUrl,
      fileName: file.name,
      size: file.size,
      mimeType: file.mime_type
    });
  } catch (error) {
    next(error);
  }
});

// Update share link
router.patch('/:id', authenticate, validateUUID('id'), validateShareLink, logFileActivity('update_share'), async (req, res, next) => {
  try {
    const { permissions, expiresAt, password, maxDownloads, isActive } = req.body;

    const shareLink = await ShareLink.findOne({
      where: {
        id: req.params.id,
        created_by: req.user.id
      },
      include: ['file']
    });

    if (!shareLink) {
      throw new NotFoundError('Share link');
    }

    // Update fields
    if (permissions) shareLink.permissions = permissions;
    if (expiresAt !== undefined) shareLink.expires_at = expiresAt ? new Date(expiresAt) : null;
    if (maxDownloads !== undefined) shareLink.max_downloads = maxDownloads;
    if (isActive !== undefined) shareLink.is_active = isActive;

    // Update password if provided
    if (password !== undefined) {
      shareLink.password = password ? await bcrypt.hash(password, 12) : null;
    }

    await shareLink.save();

    logger.info(`Share link updated: ${shareLink.token} by user ${req.user.email}`);

    res.json({
      message: 'Share link updated successfully',
      shareLink: {
        id: shareLink.id,
        token: shareLink.token,
        url: shareLink.getPublicUrl(),
        permissions: shareLink.permissions,
        expiresAt: shareLink.expires_at,
        hasPassword: !!shareLink.password,
        maxDownloads: shareLink.max_downloads,
        downloadCount: shareLink.download_count,
        isActive: shareLink.is_active,
        updatedAt: shareLink.updated_at
      }
    });
  } catch (error) {
    next(error);
  }
});

// Delete share link
router.delete('/:id', authenticate, validateUUID('id'), logFileActivity('delete_share'), async (req, res, next) => {
  try {
    const shareLink = await ShareLink.findOne({
      where: {
        id: req.params.id,
        created_by: req.user.id
      },
      include: ['file']
    });

    if (!shareLink) {
      throw new NotFoundError('Share link');
    }

    // Deactivate share link
    shareLink.is_active = false;
    await shareLink.save();

    // Check if file has any other active share links
    const otherActiveShares = await ShareLink.count({
      where: {
        file_id: shareLink.file_id,
        is_active: true,
        id: { [ShareLink.sequelize.Sequelize.Op.ne]: shareLink.id }
      }
    });

    // Update file shared status if no other active shares
    if (otherActiveShares === 0) {
      const file = await File.findByPk(shareLink.file_id);
      if (file) {
        file.is_shared = false;
        await file.save();
      }
    }

    logger.info(`Share link deleted: ${shareLink.token} by user ${req.user.email}`);

    res.json({
      message: 'Share link deleted successfully'
    });
  } catch (error) {
    next(error);
  }
});

export default router;