import { DataTypes } from 'sequelize';
import { v4 as uuidv4 } from 'uuid';
import { sequelize } from '../config/database.js';

export const ShareLink = sequelize.define('share_links', {
  id: {
    type: DataTypes.UUID,
    defaultValue: DataTypes.UUIDV4,
    primaryKey: true
  },
  file_id: {
    type: DataTypes.UUID,
    allowNull: false,
    references: {
      model: 'files',
      key: 'id'
    }
  },
  token: {
    type: DataTypes.STRING,
    allowNull: false,
    unique: true,
    defaultValue: () => uuidv4().replace(/-/g, '')
  },
  password: {
    type: DataTypes.STRING,
    allowNull: true
  },
  expires_at: {
    type: DataTypes.DATE,
    allowNull: true
  },
  permissions: {
    type: DataTypes.JSONB,
    defaultValue: ['view']
  },
  download_count: {
    type: DataTypes.INTEGER,
    defaultValue: 0
  },
  max_downloads: {
    type: DataTypes.INTEGER,
    allowNull: true
  },
  is_active: {
    type: DataTypes.BOOLEAN,
    defaultValue: true
  },
  created_by: {
    type: DataTypes.UUID,
    allowNull: false,
    references: {
      model: 'users',
      key: 'id'
    }
  },
  last_accessed: {
    type: DataTypes.DATE,
    allowNull: true
  },
  access_count: {
    type: DataTypes.INTEGER,
    defaultValue: 0
  }
}, {
  indexes: [
    {
      fields: ['token']
    },
    {
      fields: ['file_id']
    },
    {
      fields: ['created_by']
    },
    {
      fields: ['is_active']
    }
  ]
});

// Instance methods
ShareLink.prototype.isExpired = function() {
  return this.expires_at && new Date() > this.expires_at;
};

ShareLink.prototype.isDownloadLimitReached = function() {
  return this.max_downloads && this.download_count >= this.max_downloads;
};

ShareLink.prototype.canAccess = function() {
  return this.is_active && !this.isExpired() && !this.isDownloadLimitReached();
};

ShareLink.prototype.recordAccess = async function() {
  this.access_count += 1;
  this.last_accessed = new Date();
  await this.save();
};

ShareLink.prototype.recordDownload = async function() {
  this.download_count += 1;
  await this.save();
};

ShareLink.prototype.getPublicUrl = function() {
  const baseUrl = process.env.APP_URL || 'http://localhost:3000';
  return `${baseUrl}/share/${this.token}`;
};

// Class methods
ShareLink.findByToken = function(token) {
  return this.findOne({ 
    where: { 
      token,
      is_active: true 
    },
    include: ['file', 'creator']
  });
};

ShareLink.findByFile = function(fileId) {
  return this.findAll({
    where: { file_id: fileId },
    order: [['created_at', 'DESC']]
  });
};

ShareLink.findByUser = function(userId) {
  return this.findAll({
    where: { created_by: userId },
    include: ['file'],
    order: [['created_at', 'DESC']]
  });
};

ShareLink.cleanupExpired = async function() {
  const expiredLinks = await this.findAll({
    where: {
      expires_at: {
        [sequelize.Sequelize.Op.lt]: new Date()
      },
      is_active: true
    }
  });
  
  await this.update(
    { is_active: false },
    {
      where: {
        expires_at: {
          [sequelize.Sequelize.Op.lt]: new Date()
        },
        is_active: true
      }
    }
  );
  
  return expiredLinks.length;
};