import { DataTypes } from 'sequelize';
import { sequelize } from '../config/database.js';

export const Notification = sequelize.define('notifications', {
  id: {
    type: DataTypes.UUID,
    defaultValue: DataTypes.UUIDV4,
    primaryKey: true
  },
  user_id: {
    type: DataTypes.UUID,
    allowNull: false,
    references: {
      model: 'users',
      key: 'id'
    }
  },
  title: {
    type: DataTypes.STRING,
    allowNull: false
  },
  message: {
    type: DataTypes.TEXT,
    allowNull: false
  },
  type: {
    type: DataTypes.ENUM('info', 'success', 'warning', 'error'),
    defaultValue: 'info'
  },
  is_read: {
    type: DataTypes.BOOLEAN,
    defaultValue: false
  },
  data: {
    type: DataTypes.JSONB,
    defaultValue: {}
  },
  expires_at: {
    type: DataTypes.DATE,
    allowNull: true
  }
}, {
  indexes: [
    {
      fields: ['user_id']
    },
    {
      fields: ['is_read']
    },
    {
      fields: ['type']
    },
    {
      fields: ['created_at']
    }
  ]
});

// Instance methods
Notification.prototype.markAsRead = async function() {
  this.is_read = true;
  await this.save();
};

// Class methods
Notification.createNotification = async function(data) {
  return this.create({
    user_id: data.userId,
    title: data.title,
    message: data.message,
    type: data.type || 'info',
    data: data.data || {},
    expires_at: data.expiresAt
  });
};

Notification.findByUser = function(userId, includeRead = false) {
  const where = { user_id: userId };
  if (!includeRead) {
    where.is_read = false;
  }
  
  return this.findAll({
    where,
    order: [['created_at', 'DESC']]
  });
};

Notification.markAllAsRead = async function(userId) {
  return this.update(
    { is_read: true },
    { where: { user_id: userId, is_read: false } }
  );
};

Notification.cleanupExpired = async function() {
  const expiredCount = await this.destroy({
    where: {
      expires_at: {
        [sequelize.Sequelize.Op.lt]: new Date()
      }
    }
  });
  
  return expiredCount;
};

Notification.getUnreadCount = async function(userId) {
  return this.count({
    where: {
      user_id: userId,
      is_read: false
    }
  });
};