import { DataTypes } from 'sequelize';
import { sequelize } from '../config/database.js';

export const FileVersion = sequelize.define('file_versions', {
  id: {
    type: DataTypes.UUID,
    defaultValue: DataTypes.UUIDV4,
    primaryKey: true
  },
  file_id: {
    type: DataTypes.UUID,
    allowNull: false,
    references: {
      model: 'files',
      key: 'id'
    }
  },
  version: {
    type: DataTypes.INTEGER,
    allowNull: false
  },
  size: {
    type: DataTypes.BIGINT,
    allowNull: false
  },
  s3_key: {
    type: DataTypes.STRING,
    allowNull: false
  },
  s3_version_id: {
    type: DataTypes.STRING,
    allowNull: true
  },
  checksum: {
    type: DataTypes.STRING,
    allowNull: true
  },
  mime_type: {
    type: DataTypes.STRING,
    allowNull: true
  },
  is_latest: {
    type: DataTypes.BOOLEAN,
    defaultValue: false
  },
  created_by: {
    type: DataTypes.UUID,
    allowNull: false,
    references: {
      model: 'users',
      key: 'id'
    }
  },
  metadata: {
    type: DataTypes.JSONB,
    defaultValue: {}
  }
}, {
  indexes: [
    {
      fields: ['file_id']
    },
    {
      fields: ['version']
    },
    {
      fields: ['is_latest']
    }
  ]
});

// Class methods
FileVersion.findByFile = function(fileId) {
  return this.findAll({
    where: { file_id: fileId },
    order: [['version', 'DESC']]
  });
};

FileVersion.findLatest = function(fileId) {
  return this.findOne({
    where: { 
      file_id: fileId,
      is_latest: true 
    }
  });
};

FileVersion.createVersion = async function(fileId, versionData) {
  // Get the latest version number
  const latestVersion = await this.findOne({
    where: { file_id: fileId },
    order: [['version', 'DESC']]
  });
  
  const nextVersion = latestVersion ? latestVersion.version + 1 : 1;
  
  // Mark all previous versions as not latest
  await this.update(
    { is_latest: false },
    { where: { file_id: fileId } }
  );
  
  // Create new version
  return this.create({
    ...versionData,
    file_id: fileId,
    version: nextVersion,
    is_latest: true
  });
};