import { DataTypes } from 'sequelize';
import { sequelize } from '../config/database.js';

export const File = sequelize.define('files', {
  id: {
    type: DataTypes.UUID,
    defaultValue: DataTypes.UUIDV4,
    primaryKey: true
  },
  name: {
    type: DataTypes.STRING,
    allowNull: false,
    validate: {
      len: [1, 255]
    }
  },
  type: {
    type: DataTypes.ENUM('file', 'folder'),
    allowNull: false
  },
  size: {
    type: DataTypes.BIGINT,
    allowNull: true,
    defaultValue: 0
  },
  mime_type: {
    type: DataTypes.STRING,
    allowNull: true
  },
  path: {
    type: DataTypes.STRING,
    allowNull: false
  },
  s3_key: {
    type: DataTypes.STRING,
    allowNull: true
  },
  s3_version_id: {
    type: DataTypes.STRING,
    allowNull: true
  },
  parent_id: {
    type: DataTypes.UUID,
    allowNull: true,
    references: {
      model: 'files',
      key: 'id'
    },
    onDelete: 'CASCADE',
    onUpdate: 'CASCADE'
  },
  owner_id: {
    type: DataTypes.UUID,
    allowNull: false,
    references: {
      model: 'users',
      key: 'id'
    }
  },
  is_shared: {
    type: DataTypes.BOOLEAN,
    defaultValue: false
  },
  is_starred: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: false
  },
  is_deleted: {
    type: DataTypes.BOOLEAN,
    defaultValue: false
  },
  deleted_at: {
    type: DataTypes.DATE,
    allowNull: true
  },
  permissions: {
    type: DataTypes.JSONB,
    defaultValue: ['read', 'write', 'delete']
  },
  metadata: {
    type: DataTypes.JSONB,
    defaultValue: {}
  },
  checksum: {
    type: DataTypes.STRING,
    allowNull: true
  },
  thumbnail_url: {
    type: DataTypes.STRING,
    allowNull: true
  }
}, {
  indexes: [
    {
      fields: ['owner_id']
    },
    {
      fields: ['parent_id']
    },
    {
      fields: ['path']
    },
    {
      fields: ['is_deleted']
    },
    {
      fields: ['type']
    }
  ]
});

// Instance methods
File.prototype.getFullPath = function() {
  return this.path;
};

File.prototype.isFolder = function() {
  return this.type === 'folder';
};

File.prototype.isFile = function() {
  return this.type === 'file';
};

File.prototype.softDelete = async function() {
  this.is_deleted = true;
  this.deleted_at = new Date();
  await this.save();
};

File.prototype.restore = async function() {
  this.is_deleted = false;
  this.deleted_at = null;
  await this.save();
};

File.prototype.generateS3Key = function() {
  return `users/${this.owner_id}/files/${this.id}/${this.name}`;
};

// Class methods
File.findByPath = function(path, ownerId) {
  return this.findOne({ 
    where: { 
      path, 
      owner_id: ownerId,
      is_deleted: false 
    } 
  });
};

File.findByOwner = function(ownerId, includeDeleted = false) {
  const where = { owner_id: ownerId };
  if (!includeDeleted) {
    where.is_deleted = false;
  }
  return this.findAll({ where });
};

File.findInFolder = function(parentId, ownerId) {
  return this.findAll({
    where: {
      parent_id: parentId,
      owner_id: ownerId,
      is_deleted: false
    },
    order: [['type', 'DESC'], ['name', 'ASC']]
  });
};

File.findDeleted = function(ownerId) {
  return this.findAll({
    where: {
      owner_id: ownerId,
      is_deleted: true
    },
    order: [['deleted_at', 'DESC']]
  });
};

File.findShared = function(ownerId) {
  return this.findAll({
    where: {
      owner_id: ownerId,
      is_shared: true,
      is_deleted: false
    }
  });
};

File.findStarred = function(ownerId) {
  return this.findAll({
    where: {
      owner_id: ownerId,
      is_starred: true,
      is_deleted: false
    }
  });
};

File.searchFiles = function(ownerId, query) {
  return this.findAll({
    where: {
      owner_id: ownerId,
      is_deleted: false,
      name: {
        [sequelize.Sequelize.Op.iLike]: `%${query}%`
      }
    },
    order: [['updated_at', 'DESC']]
  });
};