import jwt from 'jsonwebtoken';
import { User, UserSession } from '../models/index.js';
import { logger } from '../utils/logger.js';

export const authenticate = async (req, res, next) => {
  try {
    const authHeader = req.headers.authorization;
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      logger.warn('Auth failed: No Authorization header');
      return res.status(401).json({ error: 'Access token required' });
    }
    const token = authHeader.substring(7);
    let decoded;
    try {
      decoded = jwt.verify(token, process.env.JWT_SECRET);
    } catch (err) {
      logger.warn('Auth failed: Invalid or expired JWT', err);
      if (err.name === 'TokenExpiredError') {
        return res.status(401).json({ error: 'Token expired' });
      }
      return res.status(401).json({ error: 'Invalid token' });
    }
    const session = await UserSession.findByToken(token);
    if (!session) {
      logger.warn('Auth failed: Session not found or inactive for token', { token });
      return res.status(401).json({ error: 'Invalid or expired token' });
    }
    if (session.isExpired()) {
      logger.warn('Auth failed: Session expired', { token, expires_at: session.expires_at });
      return res.status(401).json({ error: 'Session expired' });
    }
    const user = await User.findByPk(decoded.userId);
    if (!user) {
      logger.warn('Auth failed: User not found', { userId: decoded.userId });
      return res.status(401).json({ error: 'User not found' });
    }
    if (!user.is_active) {
      logger.warn('Auth failed: User inactive', { userId: decoded.userId });
      return res.status(401).json({ error: 'User inactive' });
    }
    await session.updateActivity();
    req.user = user;
    req.session = session;
    next();
  } catch (error) {
    logger.error('Authentication error:', error);
    return res.status(500).json({ error: 'Authentication failed' });
  }
};

export const authorize = (...roles) => {
  return (req, res, next) => {
    if (!req.user) {
      return res.status(401).json({ error: 'Authentication required' });
    }
    
    if (!roles.includes(req.user.role)) {
      return res.status(403).json({ error: 'Insufficient permissions' });
    }
    
    next();
  };
};

export const optionalAuth = async (req, res, next) => {
  try {
    const authHeader = req.headers.authorization;
    
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return next();
    }
    
    const token = authHeader.substring(7);
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    const session = await UserSession.findByToken(token);
    if (session && !session.isExpired()) {
      const user = await User.findByPk(decoded.userId);
      if (user && user.is_active) {
        req.user = user;
        req.session = session;
        await session.updateActivity();
      }
    }
    
    next();
  } catch (error) {
    // Ignore authentication errors for optional auth
    next();
  }
};

export const requireOwnership = (resourceParam = 'id') => {
  return async (req, res, next) => {
    try {
      const resourceId = req.params[resourceParam];
      
      // Admin users can access any resource
      if (req.user.role === 'admin') {
        return next();
      }
      
      // Check if user owns the resource (implementation depends on resource type)
      // This is a generic check - specific routes should implement their own ownership checks
      next();
    } catch (error) {
      logger.error('Ownership check error:', error);
      return res.status(500).json({ error: 'Authorization check failed' });
    }
  };
};